/*
 * MakeTIB.
 * Copyright (C) 2003, 2004, 2005 Patrick Pelissier
 *
 * This program is free software ; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307 USA 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

typedef struct {
unsigned char *code;
unsigned long size;
enum {TI92PLUS=1, TI89=3, V200=8, TITANIUM=9} output;
} obj;

/* These are the hunk IDs for A68K */
#define HunkCode 1001
#define HunkExt  1007
#define HunkEnd  1010
#define TypeEnd     0
#define TypeExport  1


#define EXTRA_INFO(x)
#define ERROR(x) {printf(x); exit(1); }
#define ERROR1(x,y) {printf(x,y); exit(1); }
#define ERROR2(x,y,z) {printf(x,y,z); exit(1); }

void removeExt(const char *source, char *dest)
{ 
  int i = 0; 
  while (source[i]!=0 & source[i]!='.') 
    dest[i++] = source[i]; 
  dest[i] = 0; 
}

long ReadDWord(FILE *fp)
{ 
  long v=fgetc(fp)<<24; 
  v|=fgetc(fp)<<16; 
  v|=fgetc(fp)<<8; 
  v|=fgetc(fp); 
  return v; 
}

void WriteDWord(FILE *fp, long x)
{
  fputc(x>>24, fp); 
  fputc((x>>16)&0xFF, fp); 
  fputc((x>>8)&0xFF, fp); 
  fputc(x&0xFF, fp);
}

void WriteDWordRAM(unsigned char *a, unsigned long x)
{ 
  *a++ = x>>24; 
  *a++ = (x>>16)&0xFF; 
  *a++ = (x>>8)&0xFF; 
  *a++ = x&0xFF;
}

void WriteWord(FILE *fp, long x)
{ 
  fputc((x>>8)&0xFF, fp);
  fputc(x&0xFF, fp);
}

void WriteByte(FILE *fp, long x)
{ 
  fputc(x, fp);
}

void WriteLong(FILE *fp, long x)
{ 
  fputc(x>>24, fp); 
  fputc((x>>16)&0xFF, fp); 
  fputc((x>>8)&0xFF, fp); 
  fputc(x&0xFF, fp);
}

obj	
ReadAmigaObj(const char *filename)
{
   obj o = {NULL, 0, 0};
   unsigned int size, type, i, ofs;
   char sym[100];
   FILE *fp = fopen(filename, "rb");
   if (fp == NULL) return o;
   
   /* Process each hunk in the object file */
   while (!feof(fp))
     {
       int hunk = ReadDWord (fp);
       if (hunk == HunkCode)
	 {
	   if (o.code) 
	     ERROR("Duplicate CODE section");
	   o.size = ReadDWord (fp) * 4; /* Size is in DWORDs */
	   o.code = malloc (o.size);
	   if (o.code == NULL)
	     ERROR("Out of Memory");
	   fread (o.code, o.size, 1, fp);
	 }
       else if (hunk==HunkExt) 	/* Exports references */
	 while (1)
	   {
	     size = ReadDWord(fp);
	     /* High byte of size contains type byte */
	     type = (size>>24)&0xff;
	     size &= 0xffffff;
	     if (type == TypeEnd)
	       break;
	     /* Read name of symbol */
	     for (i = 0 ; i < (size<<2) ; i++)
	       sym[i] = fgetc(fp);
	     sym[i]=0;
	     if (type==TypeExport)
	       {
		 /* Get offset of exported function (Useless) */
		 ofs = ReadDWord(fp);
		 /* Check for special export names */
		 if (strcmp (sym,"_ti89") == 0)
		   o.output = TI89;
		 else if (strcmp (sym,"_ti92plus") == 0)
		   o.output = TI92PLUS;
		 else if (strcmp (sym,"_v200") == 0)
		   o.output = V200;
		 else if (strcmp (sym, "_ti89ti") == 0)
		   o.output = TITANIUM;
	       } 
	     else
	       ERROR2 ("Unsupported Type: %d for symbol: %s", type, sym);
	   }
       else if (hunk==HunkEnd) /* End of code or BSS section */
	 break;
       else
	 {
	   size = ReadDWord(fp); /* Size is in DWORDs */
	   fseek (fp, size*4, SEEK_CUR);
	 }
     }
   
   fclose(fp);
   return o;
}


int	
WriteTIB2 (const obj o, const char *const filename)
{
  static const char TIBProductName[] = "Advanced Mathematics Software";
  static const char TIBSignature1[64] = {};
  FILE	*f;

  if (o.code == NULL)	
    return -1;
  if (o.size&1)		
    return -2;
  f = fopen(filename, "wb");
  if (f == NULL)	
    return -3;
  
  WriteWord(f, 0x800F); WriteLong(f, o.size + 126+8);
  WriteWord(f, 0x8011); WriteByte(f, o.output);
  WriteWord(f, 0x8021); WriteByte(f, 0x03);
  WriteWord(f, 0x8032); WriteWord(f, 0x384F);
  WriteWord(f, 0x80A1); WriteByte(f, 0x02); /* Support for HW3 */
  WriteWord(f, 0x804D); WriteByte(f, sizeof(TIBProductName)-1); fwrite(TIBProductName, sizeof(TIBProductName)-1, 1, f);
  WriteWord(f, 0x0326); WriteLong(f, 0x01020304); WriteWord(f, 0x0506);
  WriteWord(f, 0x020D); WriteByte(f, 0x40); fwrite(TIBSignature1, 64, 1, f);
  WriteWord(f, 0x807F); WriteLong(f, o.size + 8); 
  WriteLong(f, 0xCCCCCCCC);
  fwrite(o.code, o.size, 1, f);
  WriteLong(f, 0x00000000);
  WriteWord(f, 0x020D); WriteByte(f, 0x40); fwrite(TIBSignature1, 64, 1, f);
  fclose(f);
}

int	
main (int argc, const char *const argv[])
{
  obj o;
  char	Name[100];
  
  if (argc != 2)
    ERROR ("ERROR: maketib obj.o\n");
  o = ReadAmigaObj(argv[1]);
  if (o.code == NULL)
    ERROR1 ("ERROR: Bad object file %s\n", argv[1]);
  removeExt (argv[1], Name);
  strcat (Name, ".tib");
  WriteTIB2 (o, Name);
  return 0;
}
