;##################################################################
;
;   Phoenix III for the TI-86 - low level display/interrupt/background/keys
;
;   Programmed by Patrick Davidson (pad@calc.org)
;        
;   This program is in the public domain.  There is no warranty.
;
;   This file was last updated April 30, 2002.
;
;##################################################################

;############## Interrupt routine
;
; Advances grayscale frame to the next one to be displayed.  Note that the
; next frame begins display one frame after the video address port is
; updated.
;
; After a frame has been show a sufficient number of times, the interrupt
; automatically switches to the other frame, and the rest of the program
; must keep up.  This is required due to the one-frame delay, since it is
; quite difficult to predict whether the main program will finish drawing
; one frame later.
;
; Since the cycle counter is reset to 0 after the display port is set, the
; new buffer address will only be displayed on the next interrupt, after
; which the cycle is set to 1.  Due to the one-frame delay, the next buffer
; only starts being shown when the cycle is 2.

interrupt_handler:
        push    af
        push    hl

        in      a,(3)               ; Exit if not VBL interrupt
        bit     1,a
        jr      z,iexit

        ld      hl,(gs_page)        ; H = F0 or F8 (depending on buffer)
                                    ; L = 0/2/4 (page counter)
        ld      a,h
        add     a,l                 ; H = sum of buffer / page
        and     $FC                 ; turns pseudo-page 2 to 0
        out     (0),a

        ld      a,2
        add     a,h                 ; H = next page
        cp      6                   ; if page counter reaches 6

        jr      nz,no_restart
        xor     a                   ; advance to next page
no_restart:
        ld      (gs_counter),a      ; store new page

        ld      hl,num_cycles       
        inc     (hl)                ; increment grayscale cycle number
        ld      a,6                 ; continue only if at end of cycle
        cp      (hl)
        jr      nz,iexit

        xor     a                   ; restart cycle
        ld      (hl),a
        ld      a,(gs_page)         ; alternate buffer used
        xor     8
        ld      (gs_page),a

iexit:  in      a,(3)                   ; Bit 1 = ON key status
        and     1
        add     a,9                     ; A = 10 if ON pressed, 9 if not
        out     (3),a                  
        ld      a,11
        out     (3),a                   
        pop     hl
        pop     af
        ei
        reti

;############## Initialization

init:
#ifdef  __TI86__ 
        ld      hl,$CA00                ; Install interrupt
        ld      (hl),$80
        ld      bc,256
        ld      de,$CA01
        ldir

        ld      a,$ca
        ld      i,a
        im      2
#else
        ld      hl,interrupt_handler
        call    INT_INSTALL
        ret     c
#endif

        ld      hl,textShadow           ; zero variables
        ld      de,textShadow+1
        xor     a
        ld      (hl),a
        ld      bc,167
        ldir

        ld      a,$f0                   ; set up grayscale data
        ld      (gs_page),a

        ld      (exit_program+1),sp
        call    main
exit_program:
        ld      sp,0
#ifdef  __TI85__
        call    INT_REMOVE
#else
        di
        im      1
#endif
        ld      a,$3c
        out     (0),a
        call    _clrLCD
        ld      a,(game_over)
        or      a
        ret     z
        jp      do_scoring

;############## Wait for next frame, and do page swap

main_plane0:
wait_next_frame:
        ld      hl,num_cycles
        ld      a,1
lw1:    cp      (hl)
        jr      nz,lw1

        ld      a,2
lw2:    cp      (hl)
        jr      nz,lw2

        ld      a,(gs_page)
        xor     8
        ld      h,a
        ld      l,0
        ld      (gfx_buffer),hl
        dec     hl
        dec     hl
        dec     hl
        ld      (smc_gfxmem_start+1),hl
        dec     h
        dec     h
        ld      (smc_gfxmem_minus512+1),hl
        ret

;############## Display a frame from the scrolled screen

;----E800
;
; size here in noscroll_size (this is *bottom* part of screen)
;
;----scrolling line
;
; size here in scroll_size (this is *top* part of screen)
;
;----EBFF

display_scrolled:
        ld      a,(scroll_line)
        or      a
        jr      z,scroll_offset_0

        add     a,a
        add     a,a
        ld      l,a
        ld      h,0
        add     hl,hl
        add     hl,hl                   ; HL = scroll amount * 16
        ld      (scroll_size),hl

        ex      de,hl
        ld      hl,$400
        sbc     hl,de
        ld      (noscroll_size),hl

        ld      de,background_plane0+3  ; copy top of plane 0
        add     hl,de
        ld      de,(gfx_buffer)
        ld      a,(scroll_line)
        call    copy_a_lines

        ld      bc,-$400                ; copy bottom of plane 0
        add     hl,bc
        ld      a,(scroll_line)
        neg
        and     63
        call    copy_a_lines

        ld      hl,(noscroll_size)      ; copy top of plane 1
        ld      de,background_plane1+3
        add     hl,de
        ld      de,(gfx_buffer)
        set     2,d
        ld      a,(scroll_line)
        call    copy_a_lines

        ld      bc,-$400                ; copy bottom of plane 1
        add     hl,bc
        ld      a,(scroll_line)
        neg
        and     63

copy_a_lines:
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        inc     hl
        inc     hl
        inc     hl
        inc     de
        inc     de
        inc     de
        dec     a
        jr      nz,copy_a_lines
        ret

scroll_offset_0:
        ld      hl,$e803
        ld      de,(gfx_buffer)
        ld      a,64
        jr      copy_a_lines

;############## Scroll the screen up one line

scroll_screen:
        ld      a,(game_timer)
        rrca
        ret     nz
do_scroll_screen:
        ld      a,(scroll_line)
        inc     a
        and     63
        ld      (scroll_line),a

        ld      hl,ycoord
        dec     (hl)
        ld      a,(hl)
        cp      -1
        jr      nz,no_reset_map_pos

        ld      hl,(map_end_ptr)
        ld      de,-13
        add     hl,de
        ld      (map_pos),hl

no_reset_map_pos:
        and     7
        ld      c,a
        ld      de,(map_pos)
        ld      a,(scroll_line)
        call    render_background_line

        ld      a,(ycoord)
        and     7
        ret     nz
        ld      hl,(map_pos)
        ld      de,-13
        add     hl,de
        ld      (map_pos),hl
        ret

;############## Render background at (DE) into line A (segment c)

render_background_line:
        dec     a
        cpl
        add     a,a
        add     a,a
        ld      l,a
        ld      h,$3a
        inc     l
        add     hl,hl
        add     hl,hl                   ; HL = destination in back buffer
        dec     l

        ld      b,13
loop_render_background:
        ld      a,(de)                  ; A = tile #
        inc     de
        add     a,a
        add     a,a
        add     a,a
        add     a,a                     ; A = offset of this tiles
        add     a,c                     ; A = offset of this line in tile

        push    de
        push    hl
        ld      e,a
        ld      d,0
        ld      hl,(tiles_ptr)
        add     hl,de                   ; HL -> tile data
        ld      a,(hl)                  ; A = tile data (plane 0)
        ld      de,8
        add     hl,de
        ld      d,(hl)                  ; H = tile data (plane 1)
        pop     hl
        ld      (hl),a                  ; Write plane 0 data
        set     2,h
        ld      (hl),d
        res     2,h                     ; Write plane 1 data
        pop     de
        inc     hl
        djnz    loop_render_background
        ret

;############## Scroll in the top of the backgroudn

initialize_background:
        ld      hl,scroll_line
        ld      a,-64
        add     a,(hl)
        ld      (hl),a

        ld      b,64
pscr:   push    bc
        call    do_scroll_screen
        pop     bc
        djnz    pscr
        ret

;############## GET_KEY replacement

SUPER_GET_KEY:
        di
        push    hl
        push    de
        push    bc
        ld      e,0                     ; E = GET_KEY result
        ld      hl,getkeylastdata       ; HL = ptr to last read's table
        ld      a,$fe                   ; A = key port mask
        ld      c,0                     ; C = key number counter

gkol:   out     (1),a                   ; select group of keys to read
        ld      b,8                     ; B = # of keys left in group                      
        push    af                      ; save group mask

        in      a,(1)                   ; A = bit pattern of this key group
        ld      d,(hl)                  ; D = old key mask
        ld      (hl),a                  ; save this mask as old one
        cpl                             ; bit in A now 1 for pressed keys
        and     d                       ; bit in A now 1 for just-pressed

gkl:    inc     c                       ; increment # to key about to test
        rra                             ; Shift key value into carry
        jr      nc,nokey                ; NC = bit 0 = not just-pressed
        ld      e,c                     ; set E to # of key just pressed
nokey:  djnz    gkl

        pop     af                      ; restore group mask
        inc     hl                      ; move up in read table
        rlca                            ; select next mask
        cp      $7F                     ; if $7F, we are done
        jr      nz,gkol
        ld      a,e                     ; move result into A
        pop     bc
        pop     de
        pop     hl
        ei
        ret

getkeylastdata:
        .db     $ff,$ff,$ff,$ff,$ff,$ff,$ff

;############## Low-level support routines

ADD_HL_A:
        add     a,l
        ld      l,a
        ret     nc
        inc     h
        ret
