
/* Smiley's Adventure - Main program

   Copyright 2011 by Patrick Davidson.  This software may be freely
   modified and/or copied with no restrictions.  There is no warranty.

   by Patrick Davidson (pad@ocf.berkeley.edu, www.ocf.berkeley.edu/~pad/)

   updated August 20, 2011 */

#include "which.h"
#include "smiley.h"

#include <all.h>

#include <stdio.h>

#define _DEBUG__

#ifdef __DEBUG__
FILE *debug;
#endif

register game_data *w asm("a5");

#define ABS(x) ((x) > 0 ? (x) : -(x))

/*********************************** player status values */

#define ON_GROUND 0
#define FALLING 1
#define ON_LADDER 2
#define JUMPING 3
#define DEAD 4

/*********************************** screen position data */

#define MAX_ONSCR_X (SCR_WIDTH-55)
#define MAX_ONSCR_Y (SCR_HEIGHT-45)
#define MAX_Y (FG_HEIGHT<<4)

/********************* access to map, block type checking */

static void set_block(int x, int y, char n)
{
	x >>= 8;
	y >>= 8;
	if (x < 0)
		return;
	if (x >= w->fg_width)
		return;
	if (y < 0)
		return;
	if (y >= FG_HEIGHT)
		return;
	w->map[y * w->fg_width + x] = n;
}

static char get_block(int x, int y)
{
	x >>= 8;
	y >>= 8;
	if (x < 0)
		return -1;
	if (x >= w->fg_width)
		return -1;
	if (y >= FG_HEIGHT)
		return -1;
	if (y == -1)
		return 0;
	if (y < -1)
		return -1;
	return w->map[y * w->fg_width + x];
}

/*********************** change block at (x,y) to new if it was old */

static void change_block(int x, int y, int old, int new)
{
	x >>= 8;
	y >>= 8;
	if (x < 0)
		return;
	if (x >= w->fg_width)
		return;
	if (y < 0)
		return;
	if (y >= FG_HEIGHT)
		return;
	if (w->map[y * w->fg_width + x] == old)
		w->map[y * w->fg_width + x] = new;
}

/********************** returns true if block is air (can go through) */

static int is_air(int x, int y)
{
	char c = get_block(x, y);
	if (c == TILE_EMPTY) return 1;
	if (c == TILE_LADDER) return 1;
	if (c >= TILE_ARC1 && c <= TILE_ARC3) return 1;
	if (c == TILE_EXIT) return 1;
	if (c >= TILE_VARC1 && c <= TILE_VARC3) return 1;
	if (c == TILE_KEY) return 1;
	if (c == TILE_TELEPORTIN) return 1;
	if (c == TILE_TELEPORTOUT) return 1;
	if (c == TILE_GUN) return 1;
	if (c == TILE_TAKENGUN) return 1;
	if (c == TILE_LIFE) return 1;
	if (c == TILE_BREAKING) return 1;
	if (c == TILE_BROKEN) return 1;
	return (c == TILE_LADDERTOP);
}

/********************************** check what's on a line
 given (x,y) and width, tells the brick type on that line
 -1 if there are multiple non-air brick types */

static char check_line(int x, int y, int w)
{
	char a = get_block(x, y);
	char b = get_block(x + w, y);
	if (is_air(x, y)) return b;
	if (is_air(x + w, y)) return a;
	if (a == b) return a;
	return -1;
}

/************************** check vicinity of the player */

static int player_hits_top()
{
	return !(is_air(w->px, w-> py))
	       || !(is_air(w->px + (13<<4), w-> py));
}

static int player_on_air()
{
	return is_air(w->px, w->py + (14<<4))
	       && is_air(w->px + (13<<4), w->py + (14<<4));
}

static int is_ladder(int x, int y)
{
	unsigned char c;
	if ((x & 0xFF) >= 0xc0)
		return 0;
	if ((x & 0xFF) < 0x40)
		return 0;
	c = get_block(x, y);
	if (c == TILE_LADDER)
		return 1;
	if (c == TILE_LADDERBRICK)
		return 1;
	if ((c == TILE_LADDERTOP) && ((y & 0xFF) >= 0xd0))
		return 1;
	return 0;
}

static int open_side(int x)
{
	if (w->has_key)
		change_block(w->px + x, w->py + (10<<4), TILE_DOOR, TILE_EMPTY);
	return is_air(w->px + x, w->py)
	       && is_air(w->px + x, w->py + (13<<4));
}

/********************************* scroll screen if necessary */

static void check_scroll_left(void)
{
	if ((w->px>>4) < (w->x + 40)) {
		w->x = (w->px>>4) - 40;
		if (w->x < 0)
			w->x = 0;
	}
}

static void check_scroll_right(void)
{
	if (w->px>>4 > w->x + MAX_ONSCR_X) {
		w->x = (w->px>>4) - MAX_ONSCR_X;
		int MAX_X = w->fg_width << 4;
		if (w->x > MAX_X-SCR_WIDTH)
			w->x = MAX_X-SCR_WIDTH;
	}
}

static void check_scroll_up(void)
{
	if ((w->py>>4) < (w->y + 30)) {
		w->y = (w->py>>4) - 30;
		if (w->y < 0)
			w->y = 0;
	}
}

static void check_scroll_down(void)
{
	if (w->py>>4 > w->y + MAX_ONSCR_Y) {
		w->y = (w->py>>4) - MAX_ONSCR_Y;
		if (w->y > MAX_Y-SCR_HEIGHT)
			w->y = MAX_Y-SCR_HEIGHT;
	}
}

/************************************* initialize level */

static void initialize_round(void)
{
	int n, x, y;
	w->x = 0;
	w->y = 0;
	w->px = 20<<4;
	w->py = 20<<4;
	w->pstatus = ON_GROUND;
	w->pimage = 0;
	w->flying = 0;
	w->shots_remaining = 0;
	for (n = 0; n < MAX_BULLETS; w->bullets[n++].type = 0);
	for (y = 0; y < FG_HEIGHT; y++)
		for (x = 0; x < w->fg_width; x++)
			if (w->map[y * w->fg_width + x] == TILE_TAKENGUN)
				w->map[y * w->fg_width + x] = TILE_GUN;
}

static void initialize_level(void)
{
	int n, x, y, num_s, num_c;
	char c;

	w->has_key = 0;

	for (n = 0; n < MAX_SPECIALS; w->specials[n++].type = 0);
	for (n = 0; n < MAX_CREATURES; w->creatures[n++].type = 0);

	num_s = num_c = 0;
	for (x = 0; x < w->fg_width; x++) {
		for (y = 0; y < FG_HEIGHT; y++) {
			c = w->map[y * w->fg_width + x];
			if (c == TILE_TELEPORTOUT) {
				w->destx = (x << 8) + 16;
				w->desty = (y << 8) + 32;
			}
			if (num_s < MAX_SPECIALS) {
				if ((c >= TILE_ARC1 && c <= TILE_ARC3) || (c >= TILE_VARC1 && c <= TILE_VARC3))  {
					w->specials[num_s].type = 1;
					w->specials[num_s].x = x;
					w->specials[num_s].y = y;
					num_s++;
				}
			}
			if (num_c < MAX_CREATURES) {
				if (c == TILE_PORCUPINE) {
					w->creatures[num_c].type = CREATURE_PORCUPINE;
					w->creatures[num_c].x = x<<8;
					w->creatures[num_c].y = (y<<8) + (6<<4);
					w->creatures[num_c].dir = -8;
					num_c++;
					w->map[y * w->fg_width + x] = TILE_EMPTY;
				}
				if (c == TILE_SPIDER) {
					w->creatures[num_c].type = CREATURE_SPIDER;
					w->creatures[num_c].x = x<<8;
					w->creatures[num_c].y = (y<<8) + (6<<4);
					w->creatures[num_c].dir = -16;
					num_c++;
					w->map[y * w->fg_width + x] = TILE_EMPTY;
				}
				if (c == TILE_BIRD) {
					w->creatures[num_c].type = CREATURE_BIRD;
					w->creatures[num_c].x = x<<8;
					w->creatures[num_c].y = (y<<8) + (1 << 4);
					w->creatures[num_c].dir = -16;
					num_c++;
					w->map[y * w->fg_width + x] = TILE_EMPTY;
				}
			}
		}
	}

	w->frame3 = 0;
	w->timer = 0;
}

/************************************ update background */

static void update_background(void)
{
	int n, x, y;

	w->timer++;
	w->frame3++;
	if (w->frame3 == 5)
		w->frame3 = 0;

	for (n = w->frame3; n < MAX_SPECIALS; n += 3) {
		if (w->specials[n].type) {
			x = w->specials[n].x;
			y = w->specials[n].y;
			w->map[y * w->fg_width + x] += 1;
			if (w->map[y * w->fg_width + x] == TILE_ARC3 + 1)
				w->map[y * w->fg_width + x] = TILE_ARC1;
			if (w->map[y * w->fg_width + x] == TILE_VARC3 + 1)
				w->map[y * w->fg_width + x] = TILE_VARC1;
		}
	}
}

/*************************************** move creatures */

static creature *find_spare_creature(void)
{
	int n;

	for (n = 0; n < MAX_CREATURES; n++)
		if (w->creatures[n].type == CREATURE_NONE)
			return &(w->creatures[n]);

	return NULL;
}

static int get_creature_image(creature *a)
{
	switch (a->type) {
	case CREATURE_SPIDER:
		return 6;
	case CREATURE_PORCUPINE:
		return 4;
	case CREATURE_BOMB:
		return 16;
	case CREATURE_BIRD:
		return 8 + ((w->timer>>1) & 6);
	}
}

static void move_creatures(void)
{
	int n, x, y;
	creature *a;

	for (n = 0; n < MAX_CREATURES; n++) {
		x = w->creatures[n].x;
		y = w->creatures[n].y;
		switch (w->creatures[n].type) {

		case CREATURE_SPIDER:
		case CREATURE_PORCUPINE:
			Draw_Sprite((x>>4)-w->x, (y>>4)-w->y,
			            w->creatures[n].type == CREATURE_SPIDER ? 6 : 4);
			if (w->creatures[n].dir < 0) {
				if (is_air(x - 16, y + (10<<4)) || !(is_air(x - 16, y)))
					w->creatures[n].dir *= -1;
			} else {
				if (is_air(x + (18<<4), y + (10<<4)) ||
				        !(is_air(x + (18<<4), y)))
					w->creatures[n].dir *= -1;
			}
			if (Test_Collision(w->px >> 4, w->py >> 4,
			                   x>>4, y>>4, 0, 4)) {
				w->pimage = 2;
				w->pstatus = DEAD;
			}
			w->creatures[n].x += w->creatures[n].dir;
			break;

		case CREATURE_BOMB:
			if (!(is_air(x + (1<<4), y + (5<<4)))) {
				w->creatures[n].type = CREATURE_NONE;
				break;
			}
			Draw_Sprite((x>>4)-w->x, (y>>4)-w->y, 16);
			w->creatures[n].y += (w->creatures[n].dir += 3);
			if (w->creatures[n].dir < -48)
				w->creatures[n].dir = -48;
			if (Test_Collision(w->px >> 4, w->py >> 4,
			                   x>>4, y>>4, 0, 16)) {
				w->pimage = 2;
				w->pstatus = DEAD;
			}
			break;

		case CREATURE_BIRD:
			if ((w->timer & 15) == n) {
				if (ABS(x - w->px) < (10<<4)) {
					if (((w->py - y) > (10<<4)) && (y > (w->y << 4))) {
						a = find_spare_creature();
						if (a != NULL) {
							a->dir = 0;
							a->x = x + (8<<4);
							a->y = y + (8<<4);
							a->type = CREATURE_BOMB;
						}
					}
				}
			}
			Draw_Sprite((x>>4)-w->x, (y>>4)-w->y, 8 + ((w->timer>>1) & 6));
			if (w->creatures[n].dir < 0) {
				if (!(is_air(x - 16, y)))
					w->creatures[n].dir *= -1;
			} else {
				if (!(is_air(x + (18<<4), y)))
					w->creatures[n].dir *= -1;
			}
			if (Test_Collision(w->px >> 4, w->py >> 4,
			                   x>>4, y>>4, 0, 8 + ((w->timer>>1) & 6))) {
				w->pimage = 2;
				w->pstatus = DEAD;
			}
			w->creatures[n].x += w->creatures[n].dir;
			break;
		}
	}
}

/************************* non-input player movement */

static void player_move(void)
{
	int n, x, y;

	if (w->pstatus == ON_GROUND) {
		if (player_on_air()) {
			w->pstatus = FALLING;
			w->yv = 0;
		} else {
			if (check_line(w->px, w->py + (14<<4), 13<<4) == TILE_BREAKABLE) {
				w->pstatus = FALLING;
				w->yv = 0;
				change_block(w->px, w->py + (14<<4), TILE_BREAKABLE, TILE_BREAKING);
				change_block(w->px + (13<<4), w->py + (14<<4), TILE_BREAKABLE, TILE_BREAKING);
			}
		}
	}

	if (w->pstatus == JUMPING) {
		w->py += w->yv;
		w->yv += 3;
		if (player_hits_top()) {
			w->yv = 0;
			w->py = (w->py & 0xFF00) + 0x100;
		}
		if (w->yv >= 0) {
			w->pstatus = FALLING;
		}
		check_scroll_up();
		return;
	}

	if (w->pstatus == FALLING) {
		w->yv += 3;
		if (w->yv > 180) w->yv = 180;
		w->py += w->yv;
		check_scroll_down();
		if (!(player_on_air())) {
			w->py &= 0xFF00;
			w->py += 32;
			w->pstatus = ON_GROUND;
		}


		for (n = 0; n < MAX_CREATURES; n++) {
			x = w->creatures[n].x;
			y = w->creatures[n].y;
			if (w->creatures[n].type == CREATURE_SPIDER) {
				if (Test_Collision(w->px >> 4, w->py >> 4,
				                   x>>4, y>>4, 0, 4)) {
					w->creatures[n].type = CREATURE_NONE;
					w->pstatus = JUMPING;
					w->yv = -36;
				}
			}
		}

		change_block(w->px, w->py, TILE_BREAKING, TILE_BROKEN);
		change_block(w->px + (13<<4), w->py, TILE_BREAKING, TILE_BROKEN);

		if (check_line(w->px+(2<<4), w->py+(14<<4), 11<<4) == TILE_SPRING) {
			w->pstatus = JUMPING;
			w->yv = -w->yv - 10;
		}
	}
}

/************************* user input to move player */

int try_left(int dx)
{
	if (open_side(dx)) {
		w->px += dx;
		check_scroll_left();
		return 1;
	}
	if (!open_side(0))
		return 0;
	w->px = w->px & 0xff00;
	check_scroll_left();
	return 0;
}

int try_right(int dx)
{
	if (open_side(dx + (13 << 4))) {
		w->px += dx;
		check_scroll_right();
		return 1;
	}
	if (!open_side(13<<4))
		return 0;
	w->px = (((w->px + (13<<4)) & 0xff00) + 0x100) - (14<<4);
	check_scroll_right();
	return 0;
}

static void player_input(void)
{
	/* if flying, fly and respond only to slowing down */

	if (w->flying > 0)  {
		if (GETKEY(0,1,0,4))
			w->flying -= 8;
		if (!try_right(w->flying))
			w->flying = 0;
		return;
	}

	/* up arrow pressed */

	if (GETKEY(0,0,0,5)) {
		if (w->pstatus == ON_GROUND) {
			if (is_ladder(w->px + (7<<4), w->py + (13<<4))) {
				w->pstatus = ON_LADDER;
			}
		}
		if (w->pstatus == ON_LADDER) {
			if (is_ladder(w->px + (7<<4), w->py + (12<<4))) {
				w->py -= (1<<4);
			}
		}
		check_scroll_up();
	}

	/* down arrow pressed */

	if (GETKEY(0,2,0,7)) {
		if (w->pstatus == ON_GROUND) {
			if (is_ladder(w->px + (7<<4), w->py + (13<<4))) {
				w->pstatus = ON_LADDER;
			}
		}
		if (w->pstatus == ON_LADDER) {
			if (is_ladder(w->px + (7<<4), w->py + (14<<4))) {
				w->py += (1<<4);
			}
		}
		check_scroll_down();
	}

	/* left arrow pressed */

	if (GETKEY(0,1,0,4)) {
		if (w->pstatus != ON_LADDER) {
			if (!try_left(-32))
				if (get_block(w->px - 32, w->py + (2<<4)) == TILE_CANNON)
					if (get_block(w->px - 32, w->py + (11<<4)) == TILE_CANNON)
						w->flying = 8 << 4;
		} else if (is_ladder(w->px + (6<<4), w->py + (11<<4)))
			w->px -= 16;
		else if (try_left(-16))
			w->pstatus = ON_GROUND;
	}

	/* right arrow pressed */

	if (GETKEY(0,3,0,6)) {
		if (w->pstatus != ON_LADDER)
			try_right(32);
		else if (is_ladder(w->px + (8<<4), w->py + (11<<4)))
			w->px += 16;
		else if (try_right(16))
			w->pstatus = ON_GROUND;
	}

	/* jump button pressed */

	if (GETEDGE(0,4,0,3)) {
		if (w->pstatus == ON_GROUND) {
			w->pstatus = JUMPING;
			w->yv = -72;
		} else if (w->pstatus == ON_LADDER) {
			if ((player_on_air()) && !(player_hits_top())) {
				w->pstatus = JUMPING;
				w->yv = -60;
			}
		}
	}
}

/*************************** player interaction with block */

static int player_block_interact(void)
{
	char c;

	if (GETEDGE(2,5,4,3))
		return 1; /* tricheur! */

	c = get_block(w->px + (7<<4), w->py + (7<<4));
	if ((c >= TILE_ARC1 && c <= TILE_ARC3) || (c >= TILE_VARC1 && c <= TILE_VARC3)) {
		w->pstatus = DEAD;
		w->pimage = 2;
	}
	if (c == TILE_EXIT)
		return 1;
	if (c == TILE_TELEPORTIN) {
		w->pstatus = ON_GROUND;
		w->px = w->destx;
		w->py = w->desty;
		check_scroll_right();
		check_scroll_left();
		check_scroll_down();
		check_scroll_up();
	}
	if (c == TILE_KEY) {
		w->has_key = 1;
		set_block(w->px + (7<<4), w->py + (7<<4), TILE_EMPTY);
	}
	if (c == TILE_GUN) {
		w->shots_remaining += 6;
		set_block(w->px + (7<<4), w->py + (7<<4), TILE_TAKENGUN);
	}
	if (c == TILE_LIFE) {
		w->lives++;
		set_block(w->px + (7<<4), w->py + (7<<4), TILE_EMPTY);
	}
	return 0;
}

/********************************* show lives remaining */

static void show_lives(void)
{
	int x, dx, n;
	if (((w->px >> 4) + 7 - w->x) < SCR_WIDTH/2) {
		x = SCR_WIDTH - 9;
		dx = -9;
	} else {
		x = 1;
		dx = 9;
	}
	for (n = 0; n < w->lives; n++) {
		Draw_Sprite(x, SCR_HEIGHT-9, 18);
		x += dx;
	}
}

/************************************** level selection */

typedef struct {
	SYM_ENTRY *sym;
	char folder_name[9];
} var_entry;

static int tag_matches(char* x)
{
	if (*x-- != ((char)OTH_TAG)) return 0;
	if (*x-- != 0) return 0;
	if (*x-- != 'v') return 0;
	if (*x-- != 'l') return 0;
	if (*x-- != 's') return 0;
	if (*x-- != 0) return 0;
	return 1;
}

static int load_levels(void)
{
	var_entry files[200];
	int x = 0;
	char folder_name[9];
	char temp[25];
	SYM_ENTRY *item;
	unsigned short *data;
	int y;

	folder_name[9] = 0;

	item = SymFindFirst(NULL, 14);

	while ((item != NULL) && (x < 200)) {
		if (item->flags.bits.folder) {
			memcpy(folder_name, item->name, 8);
		} else {
			if (tag_matches((char *) HToESI(item->handle))) {
				data = (unsigned short *) HeapDeref(item->handle);
				if ((data[1] == 0x2C0d) && (data[2] <= 2)) {
					files[x].sym = item;
					memcpy(files[x].folder_name, folder_name, 9);
					x++;
				}
			}
		}
		item = SymFindNext();
	}

	if (x == 0) {
		ST_showHelp("No level files found (press . to exit)");
		do {
			Read_All_Keys();
		} while (!GETKEY(3,0,9,6));
		return 0;
	}

	y = 0;

	if (x > 1) {
		memset(LCD_MEM, 0, 3840);
		FontSetSys(1);

		for (y = 0; (y < 12) && (y < x); y++) {
			data = (unsigned short *) HeapDeref(files[y].sym->handle);
			sprintf(temp, "%s/%s (%d)", files[y].folder_name, files[y].sym->name, data[3]);
			DrawStr(20, 8*y, temp, A_NORMAL);
		}

		y = 0;

#ifdef __TI89__
		ST_showHelp("Choose level with up, down, 2nd");
#else
		ST_showHelp("Choose level with up, down, lock (hand)");
#endif

		DrawChar(0, 0, '>', A_NORMAL);
		do {
			Wait_Vbl();
			Read_All_Keys();
			if (GETEDGE(3,0,9,6))
				return 0;
			if ((GETEDGE(0,2,0,7)) && (y + 1 < x)) {
				DrawChar(0, 8*y, ' ', A_REPLACE);
				y += 1;
				DrawChar(0, 8*y, '>', A_REPLACE);
			}
			if ((GETEDGE(0,0,0,5)) && (y > 0)) {
				DrawChar(0, 8*y, ' ', A_REPLACE);
				y -= 1;
				DrawChar(0, 8*y, '>', A_REPLACE);
			}
		} while (!GETEDGE(0,4,0,3));
	}

	data = (unsigned short *) HeapDeref(files[y].sym->handle);
	w->num_maps = data[3];
	w->maps_type = data[2];
	if (w->maps_type == 0)
		w->maps_pointer = (char *) (&data[17]);
	else {
		w->maps_pointer = (char *) (&data[10]);
		w->eof = (char *) HToESI(files[y].sym->handle);
		w->ptr = w->maps_pointer;
		w->dat = -1;
	}
	return 1;
}

/***************************************** title screen */

static void paint_tile(int x, int y, int tile)
{
	int *s = (int *) &w->s[x + x + (y * SCR_WIDTH >> 3)];

	for (y = 0; y < 16; y++) {
		*s = (w->tiles_pointer)[tile*32 + y];
		s += SCR_WIDTH >> 4;
	}
}

static int title_screen(void)
{
	int x, c, sx = 16, d = 1;
	memset(&w->s, 0, PLANE_SIZE);
	PortSet(&w->s, SCR_WIDTH-1, SCR_HEIGHT-1);

#ifdef __TI89__
	FontSetSys(1);
	DrawStr(11, 16, "Smiley's Adventure 0.27", A_NORMAL);
	DrawStr(21, 26, "by Patrick Davidson", A_NORMAL);
	FontSetSys(0);
	DrawStr(17, 36, "pad@calc.org - http://pad.calc.org/", A_NORMAL);
	DrawStr(1, 46, "2ND: begin game", A_NORMAL);
	DrawStr(1, 53, "2ND: jump", A_NORMAL);
	DrawStr(1, 60, "UP/Down: climb ladder", A_NORMAL);
	DrawStr(85, 46, "Left/Right: move l/r", A_NORMAL);
	DrawStr(85, 53, "F1/F5: shoot l/r", A_NORMAL);
	DrawStr(85, 60, ". (dot): quit (why?)", A_NORMAL);
#else
	FontSetSys(2);
	DrawStr(24, 20, "Smiley's Adventure 0.27", A_NORMAL);
	DrawStr(38, 33, "by Patrick Davidson", A_NORMAL);
	FontSetSys(1);
	DrawStr(28, 46, "E-mail: pad@ocf.berkeley.edu", A_NORMAL);
	DrawStr(18, 56, "http://www.ocf.berkeley.edu/~pad/", A_NORMAL);
	FontSetSys(0);
	DrawStr(5, 66, "LOCK: begin game", A_NORMAL);
	DrawStr(5, 73, "LOCK: jump", A_NORMAL);
	DrawStr(5, 80, "UP/Down: go up/down ladder", A_NORMAL);
	DrawStr(125, 66, "Left/Right: move left/right", A_NORMAL);
	DrawStr(125, 73, "F1/F5: Shoot left/right", A_NORMAL);
	DrawStr(125, 80, ". (dot): Quit game (NO!!!!)", A_NORMAL);
#endif

	for (x = 0; x < (SCR_WIDTH >> 4); x++)
		paint_tile(x, SCR_HEIGHT-16, TILE_SOLID1);
	paint_tile(0, SCR_HEIGHT-32, TILE_SOLID3);
	paint_tile((SCR_WIDTH >> 4) - 1, SCR_HEIGHT-32, TILE_SOLID2);

	do {
		c++;
		for (x = c & 3; x < (SCR_WIDTH >> 4); x += 4)
			paint_tile(x, 0, TILE_ARC1 + random(3));
		for (x = 1; x < (SCR_WIDTH >> 4) - 1; x++)
			paint_tile(x, SCR_HEIGHT-32, TILE_EMPTY);
		sx += d;
		if (sx == SCR_WIDTH - 30) d = -1;
		if (sx == 16) d = 1;
		Draw_Sprite(sx, SCR_HEIGHT - 30, 0);
		Wait_Vbl();
		Display_Screen();
		Read_All_Keys();
		if (GETEDGE(3,0,9,6)) {
			PortRestore();
			return 0;
		}
	} while (!GETEDGE(0,4,0,3));

	memset(LCD_MEM, 0, 3840);
	PortRestore();
	return 1;
}

/*************************************** handle bullets */

static bullet *find_unused_bullet(void)
{
	int n;
	for (n = 0; n < MAX_BULLETS; n++)
		if (w->bullets[n].type == BULLET_NONE) {
			return &w->bullets[n];
		}
	return NULL;
}

static void fire_left(void)
{
	bullet *b;
	if ((b = find_unused_bullet())) {
		if (w->shots_remaining == 0)
			return;
		w->shots_remaining--;
		b->type = BULLET_LEFT;
		b->y = w->py + (8<<4);
		b->x = w->px - (12<<4);
	}
}

static void fire_right(void)
{
	bullet *b;
	if ((b = find_unused_bullet())) {
		if (w->shots_remaining == 0)
			return;
		w->shots_remaining--;
		b->type = BULLET_RIGHT;
		b->y = w->py + (8<<4);
		b->x = w->px + (17<<4);
	}
}

static void process_bullets(void)
{
	bullet *b = &w->bullets[0];
	int n, tile, i;
	for (n = 0; n < MAX_BULLETS; n++, b++) {
		if ((b->type == BULLET_LEFT) || (b->type == BULLET_RIGHT)) {
			if (b->type == BULLET_LEFT) {
				b->x -= 64;
				if (!(is_air(b->x, b->y + 16))) {
					b->type = BULLET_NONE;
					continue;
				}
				tile = get_block(b->x, b->y + 16);
			}
			if (b->type == BULLET_RIGHT) {
				b->x += 64;
				if (!(is_air(b->x + (10<<4), b->y + 16))) {
					b->type = BULLET_NONE;
					continue;
				}
				tile = get_block(b->x + (10<<4), b->y + 16);
			}
			if (Test_Collision(w->px >> 4, w->py >> 4,
			                   b->x >> 4, b->y >> 4, 0, 20)) {
				b->type = BULLET_NONE;
				w->pimage = 2;
				w->pstatus = DEAD;
				continue;
			}
			if (tile == TILE_TELEPORTIN) {
				b->x = w->destx + 48;
				b->y = w->desty + (b->y & 0xFF);
			}
			for (i = 0; i < MAX_CREATURES; i++)
				if (w->creatures[i].type != CREATURE_NONE)
					if (Test_Collision(b->x >> 4, b->y >> 4,
					                   w->creatures[i].x >> 4,
					                   w->creatures[i].y >> 4, 20,
					                   get_creature_image(&w->creatures[i]))) {
						w->creatures[i].type = CREATURE_NONE;
						b->type = BULLET_NONE;
						break;
					}
			Draw_Sprite((b->x >> 4) - w->x, (b->y >> 4) - w->y, 20);
		}
	}
}

/****************************************** level decoding */

static int read_bit()
{
	w->dat++;
	if (w->dat == 8) {
		w->dat = 0;
		w->ptr++;
	}
	return ((*w->ptr) >> (w->dat)) & 1;
}

static int read_bits(n)
{
	int acc = 0;
	int b;
	for (b = 0; b < n; b++)
		acc += read_bit() << b;
	return acc;
}

/****************************************** level loading

This function will read a level from the level file into our internal
array.  It returns true on success, or 0 on failure.  Failure occurs if
loading the map would read past the end of the file, would write past the
end of the map buffer, or would generate an invalid tile in the map. */

static int load_level(int n)
{
	w->level = n;

	if (w->maps_type == 0) {

		/* Version 0 maps consist of simply arrays of characters.  We load
		these maps simply by copying them.  The width of is always 32. */

		char *orig = w->maps_pointer;
		orig += n * MAP_SIZE;
		w->fg_width = 32;

		/* Copy the map byte-by-byte.  Fail if any copy would require us to
		read past EOF or generate an invalid block. */

		int o;
		for (o = 0; o < MAP_SIZE; o++) {
			if (orig > w->eof) {
				return 0;
			}
			char block = *orig++;
			if (block < 0 || block > TILE_MAX) {
				return 0;
			}
			w->map[o] = block;
		}

	} else {

		/* Version 1 maps use a simple encoding to shrink long sequences of
		zeros (empty blocks).  Where an empty block is present, the map stores
		a 1 bit followed by a 4-bit value indicating the number of zero blocks
		in a row.  For non-empty blocks, we have a 0 bit followed by 5 bits
		indicating the block value.  A 0 followed by 5 zero bits indicates the
		end of the level. */

		int ii, jj;
		int c;
		char *dest;
		dest = &w->map[0];
		while (1) {

			/* Fail if we are past EOF. */

			if (w->ptr >= w->eof) {
				return 0;
			}

			if (read_bit()) {

				/* Read a 1 bit.  This is a sequence of empty blocks. */

				jj = read_bits(4);
				while (jj >= 0) {

					/* Fail if we would write past the end of the array. */

					if (dest >= &w->map[MAP_SIZE]) {
						return 0;
					}

					*dest++ = 0;
					jj--;
				}
			} else {

				/* Read a 0 bit.  This is a non-empty block or the end. */

				jj = read_bits(5);
				if (jj == 0) break;

				/* Fail if we would write past the end of the array. */

				if (dest >= &w->map[MAP_SIZE]) {
					return 0;
				}

				/* Fail if the block value is invalid. */

				if (jj > TILE_MAX) {
					return 0;
				}

				*dest++ = jj;
			}
		}

		/* Calculate width by amount of data retrieved. */

		w->fg_width = (dest - &w->map[0]) >> 4;
	}

	return 1;
}

/***************************************** main program */

static int play_level(void)
{
	int n, x, y, num_s, num_c;
	char c;

	initialize_level();
	initialize_round();

	do {
		update_background();
		Draw_Background();
		move_creatures();
		process_bullets();
		Draw_Sprite((w->px>>4)-w->x,(w->py>>4)-w->y,w->pimage);
		show_lives();
		Wait_Vbl();
		Display_Screen();
		Read_All_Keys();
		if (GETEDGE(1,1,8,4))
			Contrast_Up();
		if (GETEDGE(1,2,9,0))
			Contrast_Down();
		if (w->pstatus != DEAD)
			if (player_block_interact())
				return 0;
		if (w->pstatus != DEAD) {
			player_input();
			player_move();
			if (w->flying >= 0)
				if (GETEDGE(5,7,6,4))
					fire_left();
			if (w->flying <= 0)
				if (GETEDGE(1,7,7,4))
					fire_right();
		}
		if (GETEDGE(4,4,5,1))
			return 0;
		if (w->pstatus == DEAD) {
            if ((GETEDGE(0,4,0,3)) || (GETEDGE(6,0,8,6)))
    		{
    			if (--w->lives == 0)
    				return 1;
    			else
    				initialize_round();
            }   
        }
	} while (!(GETKEY(3,0,9,6)));
	return 1;
}

void play_main_game(game_data *ww)
{
	int x;
	w = ww;
#ifdef __DEBUG__
	debug = fopen("debug", "w");
#endif
	if (title_screen()) {
		if (load_levels()) {
			w->lives = 3;
			for (x = 0; x < w->num_maps; x++) {
				if (!load_level(x)) {
					return;
				}

				if (play_level()) {
#ifdef __DEBUG__
					fclose(debug);
#endif
					return;
				}
			}
		}
	}
#ifdef __DEBUG__
	fclose(debug);
#endif
}
